using System;
using System.Collections.Generic;
using System.Net;
using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.Extensions.Logging;
using Polly;
using Polly.CircuitBreaker;
using Polly.Retry;
using Polly.Timeout;

namespace MatrixTrak.ShippedAssets.PollyRetryPolicies;

/// <summary>
/// Minimal, copy-paste friendly example of a resilient HTTP client.
/// - Timeout budget
/// - Exponential backoff retries w/ jitter
/// - Circuit breaker
/// - Structured logging hooks
/// 
/// Notes:
/// - This is a template: tune thresholds for your dependency.
/// - HttpRequestMessage must be cloned for retries.
/// </summary>
public sealed class ResilientHttpClient
{
    private readonly HttpClient _http;
    private readonly ILogger _log;
    private readonly ResiliencePipeline<HttpResponseMessage> _pipeline;

    public ResilientHttpClient(HttpClient http, ILogger log)
    {
        _http = http ?? throw new ArgumentNullException(nameof(http));
        _log = log ?? throw new ArgumentNullException(nameof(log));

        // Tune these per dependency.
        var maxRetries = 3;
        var timeout = TimeSpan.FromSeconds(10);
        var breakDuration = TimeSpan.FromSeconds(15);

        _pipeline = new ResiliencePipelineBuilder<HttpResponseMessage>()
            .AddTimeout(new TimeoutStrategyOptions
            {
                Timeout = timeout
            })
            .AddRetry(new RetryStrategyOptions<HttpResponseMessage>
            {
                MaxRetryAttempts = maxRetries,
                BackoffType = DelayBackoffType.Exponential,
                Delay = TimeSpan.FromSeconds(1),
                UseJitter = true,
                ShouldHandle = new PredicateBuilder<HttpResponseMessage>()
                    .Handle<HttpRequestException>()
                    .Handle<TimeoutRejectedException>()
                    .HandleResult(r => IsRetryableStatusCode(r.StatusCode)),
                OnRetry = args =>
                {
                    var status = args.Outcome.Result?.StatusCode;
                    var ex = args.Outcome.Exception;

                    _log.LogWarning(ex,
                        "HTTP retry {Attempt}/{MaxAttempts} after {DelayMs}ms (status={Status})",
                        args.AttemptNumber,
                        maxRetries,
                        (int)args.RetryDelay.TotalMilliseconds,
                        status?.ToString() ?? "(exception)");

                    return default;
                }
            })
            .AddCircuitBreaker(new CircuitBreakerStrategyOptions<HttpResponseMessage>
            {
                ShouldHandle = new PredicateBuilder<HttpResponseMessage>()
                    .Handle<HttpRequestException>()
                    .Handle<TimeoutRejectedException>()
                    .HandleResult(r => (int)r.StatusCode >= 500 || r.StatusCode == HttpStatusCode.TooManyRequests),

                // When ~50% of recent calls fail, open the breaker.
                FailureRatio = 0.5,
                SamplingDuration = TimeSpan.FromSeconds(30),
                MinimumThroughput = 20,
                BreakDuration = breakDuration,

                OnOpened = args =>
                {
                    _log.LogError(args.Outcome.Exception,
                        "Circuit opened for {BreakDurationMs}ms",
                        (int)args.BreakDuration.TotalMilliseconds);
                    return default;
                },
                OnClosed = _ =>
                {
                    _log.LogInformation("Circuit closed");
                    return default;
                },
                OnHalfOpened = _ =>
                {
                    _log.LogInformation("Circuit half-open (probing)");
                    return default;
                }
            })
            .Build();
    }

    public async Task<HttpResponseMessage> SendAsync(HttpRequestMessage request, CancellationToken ct = default)
    {
        if (request is null) throw new ArgumentNullException(nameof(request));

        // ExecuteAsync will re-run the delegate for retries, so we must clone the request each time.
        return await _pipeline.ExecuteAsync(async token =>
        {
            using var cloned = await CloneRequestAsync(request, token);
            return await _http.SendAsync(cloned, HttpCompletionOption.ResponseHeadersRead, token);
        }, ct);
    }

    private static bool IsRetryableStatusCode(HttpStatusCode code)
    {
        // Retry: 408/429 and most 5xx.
        if (code == HttpStatusCode.RequestTimeout) return true;
        if (code == (HttpStatusCode)429) return true;
        var n = (int)code;
        return n >= 500 && n <= 599;
    }

    private static async Task<HttpRequestMessage> CloneRequestAsync(HttpRequestMessage request, CancellationToken ct)
    {
        var clone = new HttpRequestMessage(request.Method, request.RequestUri)
        {
            Version = request.Version
        };

        // Copy headers
        foreach (var header in request.Headers)
        {
            clone.Headers.TryAddWithoutValidation(header.Key, header.Value);
        }

        // Copy content
        if (request.Content != null)
        {
            var ms = new System.IO.MemoryStream();
            await request.Content.CopyToAsync(ms, ct);
            ms.Position = 0;

            var contentClone = new StreamContent(ms);
            foreach (var header in request.Content.Headers)
            {
                contentClone.Headers.TryAddWithoutValidation(header.Key, header.Value);
            }

            clone.Content = contentClone;
        }

        return clone;
    }
}
