using System;
using System.Net;
using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;

namespace MatrixTrak.Production
{
    /// <summary>
    /// Minimal backpressure-aware handler for HTTP 429.
    ///
    /// This is intentionally conservative:
    /// - It honors Retry-After when present.
    /// - It caps delay.
    /// - It does not implement unbounded retries.
    ///
    /// Pair with:
    /// - per-attempt timeout
    /// - total time budget
    /// - attempt cap
    /// - idempotency guarantees
    /// </summary>
    public sealed class RetryAfterDelegatingHandler : DelegatingHandler
    {
        private readonly TimeSpan _maxDelay;
        private readonly bool _addJitter;

        public RetryAfterDelegatingHandler(TimeSpan maxDelay, bool addJitter = true)
        {
            if (maxDelay <= TimeSpan.Zero) throw new ArgumentOutOfRangeException(nameof(maxDelay));
            _maxDelay = maxDelay;
            _addJitter = addJitter;
        }

        protected override async Task<HttpResponseMessage> SendAsync(HttpRequestMessage request, CancellationToken cancellationToken)
        {
            var response = await base.SendAsync(request, cancellationToken).ConfigureAwait(false);

            if (response.StatusCode != (HttpStatusCode)429)
                return response;

            // If the upstream provides Retry-After, honor it once.
            // The caller should decide whether to retry beyond this.
            if (!TryGetRetryAfterDelay(response, DateTimeOffset.UtcNow, out var delay))
                return response;

            delay = CapDelay(delay);

            if (_addJitter)
                delay = AddJitter(delay);

            if (delay <= TimeSpan.Zero)
                return response;

            await Task.Delay(delay, cancellationToken).ConfigureAwait(false);
            return response;
        }

        private TimeSpan CapDelay(TimeSpan delay)
        {
            if (delay <= TimeSpan.Zero) return TimeSpan.Zero;
            return delay <= _maxDelay ? delay : _maxDelay;
        }

        private static TimeSpan AddJitter(TimeSpan delay)
        {
            // Small jitter to avoid synchronized retries when many instances are throttled.
            // 0-20% of delay, capped to 1s for tiny delays.
            var maxJitterMs = Math.Min(1000, (int)(delay.TotalMilliseconds * 0.2));
            if (maxJitterMs <= 0) return delay;

            var jitterMs = Random.Shared.Next(0, maxJitterMs + 1);
            return delay + TimeSpan.FromMilliseconds(jitterMs);
        }

        public static bool TryGetRetryAfterDelay(HttpResponseMessage response, DateTimeOffset now, out TimeSpan delay)
        {
            delay = TimeSpan.Zero;

            var retryAfter = response.Headers.RetryAfter;
            if (retryAfter == null) return false;

            if (retryAfter.Delta.HasValue)
            {
                delay = retryAfter.Delta.Value;
                return delay > TimeSpan.Zero;
            }

            if (retryAfter.Date.HasValue)
            {
                var target = retryAfter.Date.Value;
                delay = target > now ? (target - now) : TimeSpan.Zero;
                return delay > TimeSpan.Zero;
            }

            return false;
        }
    }
}
