using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;

namespace MatrixTrak.Production
{
    /// <summary>
    /// Minimal outbound propagation for X-Correlation-Id.
    ///
    /// - Copies X-Correlation-Id to downstream requests when present.
    /// - Does not overwrite if the request already has the header.
    ///
    /// How you get the current correlation ID depends on your platform.
    /// In ASP.NET Core, prefer passing it explicitly or using a scoped accessor.
    /// </summary>
    public sealed class CorrelationIdDelegatingHandler : DelegatingHandler
    {
        private const string HeaderName = "X-Correlation-Id";
        private readonly ICorrelationIdAccessor _accessor;

        public CorrelationIdDelegatingHandler(ICorrelationIdAccessor accessor)
        {
            _accessor = accessor;
        }

        protected override Task<HttpResponseMessage> SendAsync(HttpRequestMessage request, CancellationToken cancellationToken)
        {
            if (!request.Headers.Contains(HeaderName))
            {
                var correlationId = _accessor.GetCorrelationId();
                if (!string.IsNullOrWhiteSpace(correlationId))
                    request.Headers.TryAddWithoutValidation(HeaderName, correlationId);
            }

            return base.SendAsync(request, cancellationToken);
        }
    }

    public interface ICorrelationIdAccessor
    {
        string GetCorrelationId();
    }
}
