using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Logging;

namespace MatrixTrak.Production
{
    /// <summary>
    /// Minimal correlation ID middleware for ASP.NET Core.
    ///
    /// - Accepts inbound X-Correlation-Id or generates one.
    /// - Adds the header to the response.
    /// - Starts a logging scope so structured logs include correlation_id.
    ///
    /// Pair this with an outbound HttpClient handler that propagates the header.
    /// </summary>
    public sealed class AspNetCoreCorrelationIdMiddleware
    {
        private const string HeaderName = "X-Correlation-Id";
        private readonly RequestDelegate _next;
        private readonly ILogger<AspNetCoreCorrelationIdMiddleware> _logger;

        public AspNetCoreCorrelationIdMiddleware(RequestDelegate next, ILogger<AspNetCoreCorrelationIdMiddleware> logger)
        {
            _next = next;
            _logger = logger;
        }

        public async Task Invoke(HttpContext context)
        {
            var correlationId = GetOrCreateCorrelationId(context);

            context.Response.OnStarting(() =>
            {
                if (!context.Response.Headers.ContainsKey(HeaderName))
                    context.Response.Headers[HeaderName] = correlationId;
                return Task.CompletedTask;
            });

            using (_logger.BeginScope(new[] { new KeyValuePair<string, object>("correlation_id", correlationId) }))
            {
                await _next(context);
            }
        }

        private static string GetOrCreateCorrelationId(HttpContext context)
        {
            if (context.Request.Headers.TryGetValue(HeaderName, out var values))
            {
                var existing = values.ToString();
                if (!string.IsNullOrWhiteSpace(existing))
                    return existing;
            }

            var generated = Guid.NewGuid().ToString("N");
            context.Request.Headers[HeaderName] = generated;
            return generated;
        }
    }
}
